<?php

namespace App\Http\Controllers\Api\Auth;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Twilio\Rest\Client;
use App\Config;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Crypt;

class RegisterController extends Controller
{
    private $otpDir;
    private $config;

    public function __construct()
    {
        $this->config = Config::first();
        $this->otpDir = public_path('otp'); // Use public directory for OTP files

        if (!File::exists($this->otpDir)) {
            File::makeDirectory($this->otpDir, 0755, true);
        }
    }

    private function isTwilioEnabled()
    {
        return $this->config->otp_enable ?? false;
    }

    private function generateOTP()
    {
        return rand(100000, 999999);
    }

    private function storeOTPAndData($mobile, $otp, $data = null)
    {
        $filename = $this->otpDir . '/' . $mobile . '.txt';
        $content = $otp . "\n";
if ($data && isset($data['mobile'])) {
        // Don't encrypt data if logging in with mobile number
        $content .= json_encode($data) . "\n"; // Store data without encryption
    } else {
        // Encrypt data if it's not a mobile login
        $encryptedData = Crypt::encrypt(json_encode($data));
        $content .= $encryptedData . "\n";
    }
        if ($data) {
            $encryptedData = Crypt::encrypt(json_encode($data));
            $content .= $encryptedData . "\n";
        }

        $expirationTime = time() + 60; // 1 minute from now
        $content .= $expirationTime;
        File::put($filename, $content);

        // Start a background process to delete the file after 1 minute
        $this->startBackgroundDeletion($filename, $expirationTime);
    }

    private function startBackgroundDeletion($filename, $expirationTime)
    {
        $command = "php -r \"sleep(" . ($expirationTime - time()) . "); " .
                   "if (file_exists('$filename')) { unlink('$filename'); }\" > /dev/null 2>&1 & echo $!";
        exec($command);
    }

    private function getOTPAndData($mobile)
    {
        $filename = $this->otpDir . '/' . $mobile . '.txt';
        if (File::exists($filename)) {
            $content = File::get($filename);
            $lines = explode("\n", trim($content));

            $otp = $lines[0]; // First line is the OTP
            $data = null;
            $expirationTime = null;

            if (count($lines) > 1) {
                $data = json_decode(Crypt::decrypt($lines[1]), true); // Second line is encrypted data
            }
            if (count($lines) > 2) {
                $expirationTime = (int)$lines[2]; // Third line is expiration timestamp
            }

            // Check if OTP has expired
            if ($expirationTime && time() > $expirationTime) {
                return null; // OTP expired
            }

            return [
                'otp' => $otp,
                'data' => $data
            ];
        }

        return null; // File not found
    }

    private function deleteOTPFile($mobile)
    {
        $filename = $this->otpDir . '/' . $mobile . '.txt';
        if (File::exists($filename)) {
            File::delete($filename);
        }
    }

   public function register(Request $request)
{
    // Validation rules for registration
    $rules = [
        'name' => 'required|string|max:255',
        'email' => 'required|string|email|max:255|unique:users',
        'password' => 'required|string|min:8',
        'mobile' => 'required|string|unique:users,mobile',
    ];

    // Add captcha validation if enabled in config
    if ($this->config->captcha == 1) {
        $rules['g-recaptcha-response'] = 'required|captcha';
    }

    // Validate the request data against the rules
    $request->validate($rules);

    // Call the method to register the user directly
    return $this->registerUser($request);
}

private function registerUser($request)
{
    // Create a new user with the validated request data
    $user = User::create([
        'name' => $request->name,
        'email' => $request->email,
        'password' => Hash::make($request->password),
        'mobile' => $request->mobile,
    ]);

    // Log in the newly registered user
    Auth::login($user);

    // Create an API token for the user
    $token = $user->createToken('API Token');

    // Return a successful registration response
    return response()->json(['message' => 'Registration successful!', 'token' => $token], 201);
}


    public function resendOTP(Request $request)
    {
        $mobile = $request->input('mobile');

        if (!$mobile) {
            return response()->json(['error' => 'No phone number provided for OTP resend.'], 400);
        }

        $storedData = $this->getOTPAndData($mobile);

        if (!$storedData) {
            return response()->json(['error' => 'No existing OTP data found.'], 400);
        }

        $newOtp = $this->generateOTP();

        try {
            $accountSid = env('TWILIO_SID');
            $authToken = env('TWILIO_AUTH_TOKEN');
            $twilioNumber = env('TWILIO_NUMBER');

            $client = new Client($accountSid, $authToken);
            $client->messages->create(
                $mobile,
                [
                    'from' => $twilioNumber,
                    'body' => "Your new OTP is: $newOtp"
                ]
            );

            $this->storeOTPAndData($mobile, $newOtp, $storedData['data']);

            return response()->json(['message' => 'New OTP sent.'], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to resend OTP: ' . $e->getMessage()], 500);
        }
    }

    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required',
            'password' => 'required'
        ]);

        $authUser = User::where('email', $request->email)->first();
        
        if (!$authUser) {
            return response()->json(['message' => 'Unauthorized!'], 401);
        }

        if ($authUser->is_blocked == 1) {
            return response()->json(['message' => 'Blocked User'], 403);
        }

        if ($authUser->status == 0) {
            return response()->json(['message' => 'Please verify your email!'], 403);
        }

        if (Hash::check($request->password, $authUser->password)) {
            $token = $authUser->createToken('AuthToken')->accessToken;
            return response()->json(['access_token' => $token], 200);
        }

        return response()->json(['message' => 'Unauthorized!'], 401);
    }

    public function initiateOTPLogin(Request $request)
    {
        $request->validate([
            'mobile' => 'required|string'
        ]);

        $mobile = $request->input('mobile');
        $otp = $this->generateOTP();

        try {
            $accountSid = env('TWILIO_SID');
            $authToken = env('TWILIO_AUTH_TOKEN');
            $twilioNumber = env('TWILIO_NUMBER');

            $client = new Client($accountSid, $authToken);
            $client->messages->create(
                $mobile,
                [
                    'from' => $twilioNumber,
                    'body' => "Your OTP for login is: $otp"
                ]
            );

            $this->storeOTPAndData($mobile, $otp); // No need to store user data

            return response()->json(['message' => 'OTP sent for login. Please verify your phone number.'], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to send OTP: ' . $e->getMessage()], 500);
        }
    }

    public function verifyOTP(Request $request)
    {
        $request->validate([
            'otp' => 'required|numeric',
            'mobile' => 'required|string',
        ]);

        $storedData = $this->getOTPAndData($request->mobile);

        if (!$storedData) {
            return response()->json(['error' => 'OTP is expired or invalid.'], 400);
        }

        if ($storedData['otp'] == $request->otp) {
            $user = User::where('mobile', $request->mobile)->first();
            if ($user) {
                Auth::login($user);
                $token = $user->createToken('API Token')->accessToken;

                return response()->json(['message' => 'Login successful!','token' => $token], 200);
            }

            return response()->json(['error' => 'User not found.'], 404);
        }

        return response()->json(['error' => 'Invalid OTP.'], 400);
    }
}